package org.interactivemesh.scala.j3d.samples.charcube

// Java
import java.awt.{Color, Dimension, Font, Graphics2D, GraphicsDevice, Point}
import java.awt.image.BufferedImage

import java.io.{File, IOException}

import javax.imageio.{ImageIO, ImageWriter}
import javax.imageio.spi.ImageWriterSpi;
import javax.imageio.stream.{FileImageOutputStream, ImageOutputStream}

// Scala
import scala.math.BigDecimal

// Scala Swing
import scala.swing.{Alignment, BorderPanel, Button, ButtonGroup, BoxPanel, 
  CheckBox, Component, Dialog, FileChooser, GridPanel, Label, Frame, Menu,MenuItem, 
  Orientation, RadioButton, RootPanel, ScrollPane, Separator, Slider, Swing, Window}
import swing.event.{ButtonClicked, MouseClicked, MouseEntered, MouseExited, 
  MousePressed, MouseReleased, UIElementResized, ValueChanged}    

// ImScalaSwing API 1.0, see http://www.interactivemesh.org/testspace/j3dmeetsscala.html
import org.interactivemesh.scala.swing.PopupMenu

// ScalaCanvas3D API 1.0, see http://www.interactivemesh.org/testspace/j3dmeetsscala.html
import org.interactivemesh.scala.swing.j3d.SCanvas3D

/*
 * CharacterCubePanel.scala
 *
 * Version: 1.3
 * Date: 2011/05/26
 *
 * Copyright (c) 2010-2011
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 *
 * Please create your own implementation.
 * This source code is provided "AS IS", without warranty of any kind.
 * You are allowed to copy and use all lines you like of this source code
 * without any copyright notice,
 * but you may not modify, compile, or distribute this 'CharacterCubePanel.scala'.
 *
 */

protected final class CharacterCubePanel(
    gd: GraphicsDevice, 
    private val screenSize: Dimension,
    private[charcube] var frame: App) 
      extends BoxPanel(Orientation.Horizontal) {
	
  private val bgColor = new Color(0.0f, 0.4f, 0.8f)
  private val fgColor = new Color(1.0f, 1.0f, 1.0f)
  private val stColor = new Color(0.8f, 0.8f, 0.8f)
  
  private val screenHeight = screenSize.height

  // screenHeight >= 1200
  private var textFontSize = 18
  private var titleFontSize = 30
  private var borderBLR = 50
  private var borderT = 80
  private var space = 12
  private var dialogScale = 5

  // screenHeight  < 1024
  if (screenHeight < 1024) {
    textFontSize = 12
    titleFontSize = 22
    borderBLR = 30
    borderT = 50
    space = 8
    dialogScale = 3
  }
  // 1024 <= screenHeight < 1200
  else if (screenHeight < 1200) {
    textFontSize = 16
    titleFontSize = 26
    borderBLR = 40
    borderT = 60
    space = 10
    dialogScale = 4
  }

  private val charFont: Font = new Font("Lucida Sans", Font.PLAIN, titleFontSize)
  private val titleFont: Font = new Font("Dialog", Font.PLAIN, titleFontSize)
  private val textFont: Font = new Font("Dialog", Font.PLAIN, textFontSize)

  // Frame, not applet
  private val isFrame = frame.isInstanceOf[CharacterCubeFrame.FrameApp]
  
  // Universe / Canvas3D
  private val universe = new CharacterCubeUniverse(this, gd)
  private val canvas3d = new SCanvas3D(universe.canvas3D) {
    var ratio: Double = 0
	listenTo(this, canvasMouseClick)
	reactions += {
	  case e: UIElementResized =>
	 	val size = e.source.size
	 	ratio = size.getHeight / size.getWidth
	 	val bigRatio = BigDecimal.apply(ratio, new java.math.MathContext(4))
	 	controlPanel.ratioLabel.text = bigRatio.toString
	 	
	 	// Adjust image size
	 	if (isFrame) 
	 	  controlPanel.adjustImageSizeSlider
	 	
	 	universe.setupBgImage(size.width, size.height)
	 	
      case e: MouseClicked =>
        if (e.peer.getButton == java.awt.event.MouseEvent.BUTTON3)
          popupMenu.show(e.source, e.point.x+3, e.point.y)
        else if (e.peer.getButton == java.awt.event.MouseEvent.BUTTON1 && e.clicks == 2)
          universe.shapePicked(e.point.x, e.point.y)
	}
  }  
  
  import universe.ViewPoint._
    
  // Heavyweight popup menu
  PopupMenu.defaultLightWeightPopupEnabled = false
  
  private object popupMenu extends PopupMenu {	
	  
	border = Swing.LineBorder(Color.WHITE)
	  
    private class PopMenu(s: String) extends Menu(s) {
      peer.getPopupMenu.setBorder(Swing.LineBorder(Color.WHITE))
      opaque = true
      font = textFont
      background = bgColor
      foreground = fgColor
    }   

    private abstract class PopupMenuItem(s: String) extends MenuItem(s) {
      font = textFont
      background = bgColor
      foreground = fgColor
      
      def itemAction: Unit
      
      listenTo(mouse.clicks)
      reactions += {
      	case e: MouseReleased => itemAction
      	case _ =>
      }
    }
    
    contents += new PopMenu("Viewpoints") {
      contents += new PopupMenuItem("Arrows") {
        def itemAction = controlPanel.vantagePoint(Back)
      }
      contents += new PopupMenuItem("Dingbats") {
        def itemAction = controlPanel.vantagePoint(Front)
      }
      contents += new PopupMenuItem("Latin") {
        def itemAction = controlPanel.vantagePoint(Right)
      }
      contents += new PopupMenuItem("Math Ops") {
        def itemAction = controlPanel.vantagePoint(Left)
      }
      contents += new Separator
      contents += new PopupMenuItem("Center") {
        def itemAction = controlPanel.vantagePoint(Center)
      }
      contents += new PopupMenuItem("Home") {
        def itemAction = controlPanel.vantagePoint(Home)
      }
    }
    contents += new Separator
    contents += new PopupMenuItem("Stop rotation") {
      def itemAction = controlPanel.rotationSlider.value = 50
    }
       
    if (isFrame) {
      contents += new Separator
      contents += new PopupMenuItem("Switch window") {
    	def itemAction = frame.asInstanceOf[CharacterCubeFrame.FrameApp].switchAppFrame
      }
      contents.addSeparator
      contents += new PopupMenuItem("Exit application") {
        def itemAction = frame.asInstanceOf[CharacterCubeFrame.FrameApp].closeApp
      }
    }
  }
      
  //
  // Control panel 
  //
  private object controlPanel extends BoxPanel(Orientation.Vertical) { 
	  
	// Heavyweight tooltip component
    javax.swing.ToolTipManager.sharedInstance.setLightWeightPopupEnabled(false)

	listenTo(mouse.clicks)
	reactions += {
      case e: MouseClicked =>
        if (e.peer.getButton == java.awt.event.MouseEvent.BUTTON3)
          popupMenu.show(e.source, e.point.x+3, e.point.y)
	}
	  
    background = bgColor	      
	border = Swing.EmptyBorder(0, 0, 0, borderBLR)
	
	private var maxWidth: Int = 0;
  
    private val centerActionLabel = new SActionLabel("Center") {
      tooltip = "Center Viewpoint"
      def action = vantagePoint(Center);
    }		
    
    // Set viewpoint, stop/reset rotation
    private[CharacterCubePanel] def vantagePoint(vp: ViewPoint) {
	  universe.resetRotation
	  rotationSlider.value = 50
	  universe.vantagePoint(vp)
    }

	private[CharacterCubePanel] val rotationSlider = new Slider {
      opaque = false  
      min = 0
      max = 100
      value = 50
      xLayoutAlignment = 0.5
      tooltip = "Rotation duration = [ 2 sec  -  94 sec ]"
    	  
      listenTo(this)
      reactions += {
        case ValueChanged(_) => universe.rotationSpeed(value)
      }
    }
    
    // FPS 
    private[CharacterCubePanel] val fpsLabel = new TLabel("0") {
      horizontalAlignment = Alignment.Left
    }
	// Unicode
    private[CharacterCubePanel] val unicodeCPLabel = new TLabel("---") {
      horizontalAlignment = Alignment.Left
      tooltip = "Unicode code point"
    }
    private[CharacterCubePanel] val unicodeCharLabel = new TLabel("--") {
      font = charFont
      horizontalAlignment = Alignment.Center
      tooltip = "Unicode character"
    }
    
    contents.append(
    		
	  // Title
      new BoxPanel(Orientation.Vertical) {
    	opaque = false
    	contents.append(
          new TLabel("Java 3D meets Scala") {
            font = titleFont
            
            maxWidth = preferredSize.width
          },
          new TLabel("CharacterCube") {
            font = titleFont
          }
        )
        override def paintComponent(g2d: Graphics2D) {               
          super.paintComponent(g2d)                
          val size = this.size
          g2d.setColor(Color.WHITE)              
          // Horizontal line
          g2d.drawLine(0, size.height/2, size.width, size.height/2)
	    }
      },

      Swing.VStrut(space),
      
      // Viewpoints
      new TLabel("Viewpoints"),
      Swing.VStrut(space/2),
      new SGridPanel(3, 2) {  
	    opaque = false
        vGap = space/2
        contents += (
          new SActionLabel("Arrows") {
     	    tooltip = "Arrows Viewpoint"
            def action = vantagePoint(Back)
          },
          new SActionLabel("Dingbats") {
     	    tooltip = "Dingbats Viewpoint"
            def action = vantagePoint(Front)
          },         
          new SActionLabel("Latin") {
     	    tooltip = "Latin Viewpoint"
            def action = vantagePoint(Right)
          },
          new SActionLabel("Math Ops") {
     	    tooltip = "Mathematical Operators Viewpoint"
            def action = vantagePoint(Left)
          },				
          centerActionLabel,		
          new SActionLabel("Home") {
     	    tooltip = "Home Viewpopint"
            def action = vantagePoint(Home)
          }
		)        
		
		maximumSize = new Dimension(maxWidth, this.preferredSize.height)

	    override def paintComponent(g2d: Graphics2D) {               
          super.paintComponent(g2d)                
          val size = this.size
          g2d.setColor(Color.WHITE)              
          // Vertical line
          g2d.drawLine(size.width/2, 0, size.width/2, size.height)
          // Horizontal line
          g2d.drawLine(0, size.height/3, size.width, size.height/3)
          g2d.drawLine(0, size.height*2/3, size.width, size.height*2/3)
	    }
      },
	
      Swing.VStrut(2*space),
      
      // Projection mode
      new TLabel("Projection") {
    	tooltip = "3D projection mode"
      },
      new GridPanel(1, 2) {  
	    opaque = false
        val par = new RadioButton("Parallel") {
    	  font = textFont
    	  foreground = fgColor
    	  opaque = false
    	  tooltip = "Parallel projection mode"
    	}
        val per = new RadioButton("Perspective") {
          font = textFont
          foreground = fgColor
          opaque = false
    	  selected = true
          tooltip = "Perspective projection mode"
        }
        val mutex = new ButtonGroup(par, per)
        
        contents ++= mutex.buttons
        
        listenTo(par, per)
        reactions += {
          case ButtonClicked(`par`) => {
            universe.projectionMode(javax.media.j3d.View.PARALLEL_PROJECTION)
            centerActionLabel.enabled = false
          }
          case ButtonClicked(`per`) => {
            universe.projectionMode(javax.media.j3d.View.PERSPECTIVE_PROJECTION)
            centerActionLabel.enabled = true
          }
        }
        
        maximumSize = this.preferredSize
      },
      
      Swing.VStrut(2*space),
      
      // Rotation
      new TLabel("<   Rotation    >") {
    	tooltip = "Cube's rotation direction and speed"
      },
      Swing.VStrut(space/2),
      rotationSlider,
	
      Swing.VStrut(2*space),
      
	  // Transparency
      new TLabel("Transparency") {
    	tooltip = "Cube's transparency value"
      },
      Swing.VStrut(space/2),
      new Slider {
        opaque = false  
        min = 0
        max = 100
        value = 30
        xLayoutAlignment = 0.5
        tooltip = "Transparency range [0, 1]"

        listenTo(this)
        reactions += {
          case ValueChanged(_) => universe.cubeTransparency(value)
        }
      },    
      
      Swing.VStrut(2*space),
      
      new SGridPanel(1, 2) {
    	opaque = false
    	contents += (
          new TLabel("   Character : ") {
            horizontalAlignment = Alignment.Right
            tooltip = "Picked 3D character"
          }, 
          new GridPanel(1, 2) {
    	    opaque = false
    	    contents += (
              unicodeCPLabel, unicodeCharLabel 
            )
            maximumSize = this.preferredSize
          }
        )
        maximumSize = this.preferredSize
      },
      
      Swing.VStrut(space/2),
      
      new SGridPanel(1, 2) {
    	opaque = false
    	tooltip = "Frames per second while cube is rotating"
    	contents += (
          new TLabel("F P S : ") {
            horizontalAlignment = Alignment.Right            
          }, 
          fpsLabel
        )
        maximumSize = this.preferredSize
      },
      
      Swing.VGlue
    )
      
    //
    // If isFrame
    //
    val maxDimBuffer = universe.maxOffscreenBufferSize
    val maxSizeBuffer = Math.min(4096, Math.min(maxDimBuffer.width, maxDimBuffer.height))	
    val minSizeBuffer = 16
	var widthBuffer = 0
	var heightBuffer = 0
	  
	var transBackground = false
      
    val ratioLabel = new TLabel("0.9999") {
      horizontalAlignment = Alignment.Left
    }
  
    val imageSizeLabel = new TLabel("9999 x 9999") {
      horizontalAlignment = Alignment.Left
    }
      
	var imageSizeSlider = new Slider {
	  opaque = false  
	  min = minSizeBuffer
	  max = maxSizeBuffer
	  value = 0
	  xLayoutAlignment = 0.5
	  tooltip = "From [] to []"
	
	  listenTo(this)
	  reactions += {
	    case ValueChanged(_) => {
	      if (canvas3d.ratio > 1) {
	    	heightBuffer = value
	    	widthBuffer = (heightBuffer / canvas3d.ratio).asInstanceOf[Int]
	      }
	      else {
	    	widthBuffer = value
	    	heightBuffer = (widthBuffer * canvas3d.ratio).asInstanceOf[Int]
	      }
	      imageSizeLabel.text = widthBuffer.toString + " x " + heightBuffer.toString 
	    }
	  }
	}
    
    // When canvas3d.ratio has changed
	def adjustImageSizeSlider {
	  val v = controlPanel.imageSizeSlider.value
	  if (v == controlPanel.imageSizeSlider.min)
	    controlPanel.imageSizeSlider.value = v+1
	  else 
	    controlPanel.imageSizeSlider.value = v-1
	  controlPanel.imageSizeSlider.value = v
	  
	  if (canvas3d.ratio > 1) {
	 	imageSizeSlider.tooltip = 
	 	  "From ["+ (minSizeBuffer / canvas3d.ratio).asInstanceOf[Int] + " x " + minSizeBuffer +
	 	  "] to ["+ (maxSizeBuffer / canvas3d.ratio).asInstanceOf[Int] + " x " + maxSizeBuffer + "]"
	  }
	  else {
	    imageSizeSlider.tooltip =
	 	  "From ["+ minSizeBuffer + " x " + (minSizeBuffer * canvas3d.ratio).asInstanceOf[Int] +
	 	  "] to ["+ maxSizeBuffer + " x " + (maxSizeBuffer * canvas3d.ratio).asInstanceOf[Int] + "]"
	  }
    }

    if (isFrame) { 	   	
      contents.append(
    		  
        new SActionLabel("Render to Image") {
          tooltip = "Render 3D scene into an image"
          def action = universe.renderToImage(widthBuffer, heightBuffer, transBackground)
        },        
                        
    	Swing.VStrut(space/2),
    	
    	new GridPanel(1, 2) {
          opaque = false
          tooltip = "Current width / height ratio"
    	  contents += (
    	    new TLabel("Ratio : ") {
    	      horizontalAlignment = Alignment.Right
    	    }, 
    	    ratioLabel
    	  )	
    	  maximumSize = this.preferredSize
        },
    	Swing.VStrut(space/2),    	
    	new GridPanel(1, 2) {
          opaque = false
    	  tooltip = "Image size in pixel"
    	  contents += (
    	    new TLabel("W x H : ") {
    	      horizontalAlignment = Alignment.Right
    	    }, 
    	    imageSizeLabel
    	  )	
    	  maximumSize = this.preferredSize
        },
        
    	Swing.VStrut(space/2),    
    	
    	imageSizeSlider,      
    	
    	Swing.VStrut(space/2),
    	
        // Transparent background
        new CheckBox("Transparent Background") {
          font = textFont
          foreground = fgColor
          opaque = false
          tooltip = "Transparent background while rendering to image"
          xLayoutAlignment = 0.5
        
          listenTo(this)
          reactions += {
            case ButtonClicked(_) => {
              transBackground = !transBackground
            }
          }
        }
    	
      )
      
    }
  } // End of controlPanel

  //
  // Application panel
  //
	
  listenTo(mouse.clicks)
  reactions += {
    case e: MouseClicked =>
      if (e.peer.getButton == java.awt.event.MouseEvent.BUTTON3)
        popupMenu.show(e.source, e.point.x+3, e.point.y)
  }
	  
  background = bgColor
  border = Swing.EmptyBorder(borderBLR, borderBLR, borderBLR, borderBLR)

  contents.append(controlPanel, canvas3d)
  
  // Print properties
  frame.printSystemProps
  universe.printJava3DProps
     
  // GUI update
  
  // Frames per second
  private[charcube] def updateFPS(fps: Int) = controlPanel.fpsLabel.text = fps.toString  
  // Unicode
  private[charcube] def unicodeCodePointSelected(u: Int) {
	Swing.onEDT({
	  controlPanel.unicodeCPLabel.text = u.toString
	  controlPanel.unicodeCharLabel.text = new String(Character.toChars(u))
	})
  }
  
  // Cleanup
  
  private[charcube] def closePanel {
	universe.closeUniverse
  }
  
  //
  // Show and save image
  //
  
  private var chooserDim: Dimension = new Dimension(100*(dialogScale+1), 100*dialogScale)
  private var lastSaveDir: File = null
  private var imageWriterSpi: ImageWriterSpi = null
	  
  private val maxDialogHeight = (screenSize.height * 0.8).asInstanceOf[Int]
  private val maxDialogWidth = Math.min((maxDialogHeight * 1.4).asInstanceOf[Int], screenSize.width)
	
  import FileChooser.Result._
  
  // Image dialog, called from universe
  private[charcube] def showImageDialog(image: BufferedImage) {
	  
    if (imageWriterSpi eq null) {
      val writerIterator = ImageIO.getImageWritersByFormatName("png") 
      if (writerIterator ne null) {
        while (writerIterator.hasNext && (imageWriterSpi eq null))
          imageWriterSpi = writerIterator.next.getOriginatingProvider
      }      	
    }
	   
	Swing.onEDT({
		
	    val dialog = new ImageDialog(frame.asInstanceOf[Frame]) { outer => 
	
		  title = "3D Scene : Rendered to Image"  		
		  modal = true
		   	   
		  val imageView = new ImageView(image)
		  val scrollPane = new ScrollPane(imageView)
		   
		  contents = new BorderPanel { panel => 
		   
		    layout(scrollPane) = BorderPanel.Position.Center
		    
		    if (imageWriterSpi eq null) {	    	
		      layout(
		        new SLabel("  Sorry, no 'png'-image-writer available !  ") {
		          foreground = Color.BLACK
		        }
		      ) = BorderPanel.Position.South
		    }
		    else {	    	
		      layout(
		        new Button("Save Image") {
	              reactions += {
	                case ButtonClicked(_) => {
	            	  
	            	  val fileChooser = new FileChooser(lastSaveDir)
	            	  fileChooser.title = "Save Image (No warning when overwriting existing file !!)"
	            	  fileChooser.peer.setPreferredSize(chooserDim)       // missing in Scala FileChooser ?
	            	  fileChooser.peer.addChoosableFileFilter(pngFilter)  // missing in Scala FileChooser ?
	        	      fileChooser.fileFilter = pngFilter
	
	            	  val result = fileChooser.showSaveDialog(panel)
	            	  
	            	  result match {
	            	    case Approve if (fileChooser.fileFilter eq pngFilter) =>
	            	 	              		            	      
	            	      var selFile = fileChooser.selectedFile
	            	      lastSaveDir = selFile.getParentFile
	            	      
	            	      var savePath = selFile.getPath
	            	      
	            	      if (!savePath.endsWith(".png")) {
	            	        if (savePath.endsWith("."))
	            	     	  savePath += "png"
	            	     	else
	            	     	  savePath += ".png"            	     	
	            	      }
	            	       
	            	      if (selFile.exists) selFile.delete
	            	      
	            	      selFile = new File(savePath)
	            		
	            	      val imageWriter = imageWriterSpi.createWriterInstance
	            	      val imageStream = new FileImageOutputStream(selFile)
	            	      imageWriter.setOutput(imageStream)
	            	      imageWriter.write(image)
	
	            	      imageStream.close
	            	      imageWriter.reset
	            	      imageWriter.dispose
	            	  
	            	      outer.dispose
	            	      
	            	    case _ => println("Save Image: Nothing saved !!!!!!!!!!!!!!!!!!!!!!")            	                  	       
	            	  }
	            	   
	            	  chooserDim = fileChooser.peer.getSize
	                }
	              }
	            }	    
		      ) = BorderPanel.Position.South
		    
		    }
		  }
		
		  this.pack
			
		  private val frameWidth = if (size.width < maxDialogWidth) size.width else maxDialogWidth
		  private val frameHeight = if (size.height < maxDialogHeight) size.height else maxDialogHeight
		 	
		  // Frame location and size 
		  this.location = new Point((screenSize.width - frameWidth)/2, (screenSize.height - frameHeight)/2)
		  this.size = new Dimension(frameWidth, frameHeight);
	
		  // Not called from default Scala Dialog implementation !? See ImageDialog below
		  override def closeOperation() {
			// Free image instance for gc
		 	scrollPane.contents = new Label("")	 	
	        imageView.clear
	        
	        System.gc
	      }	
	    }
	    
	    dialog.visible = true
    })
  }
  
  //
  // Nested GUI classes, objects 
  //
  
  // Lightweight 'buttons'
  private[CharacterCubePanel] class SLabel(s: String) extends Label(s) {
    def this() = this("")
    
    protected val enterColor = Color.ORANGE
    protected val pressColor = Color.RED       
    protected val selectColor = new Color(0.0f, 0.8f, 0.8f)  //(0.0f, 1.0f, 0.6f)            
	private   var isPressed = false
	private   var isSelected = false
        
    font = textFont
    foreground = fgColor
    xLayoutAlignment = 0.5
	 
    protected def pressed: Boolean = isPressed
    protected def pressed_=(p: Boolean) = isPressed = p

    protected def selected: Boolean = isSelected
    protected def selected_=(s: Boolean) = isSelected = s

	protected def contains(p: Point): Boolean = {
	  peer.contains(p.x, p.y)
	}
  }
  
  private[CharacterCubePanel] class TLabel(s: String) extends SLabel(s) {
	def this() = this("")
	
	foreground = stColor
  }

  private[CharacterCubePanel] abstract class SActionLabel(s: String) extends SLabel(s) {
	def this() = this("")
	
    listenTo(mouse.clicks, mouse.moves)
	reactions += {	 	  
	  case e: MouseEntered =>
	    if (!pressed) foreground = enterColor
	  case e: MouseExited =>
	    if (!pressed) foreground = fgColor
	  case e: MousePressed =>
	    pressed = true
        foreground = pressColor
	  case e: MouseReleased =>
        pressed = false        
        if (contains(e.point)) {
        	foreground = fgColor // enterColor 
        	action
        }
        else {
        	foreground = fgColor
        }
	}
    
    def action
  }

  // GridPanel to paint on
  private[CharacterCubePanel] class SGridPanel(rows: Int, cols: Int) extends GridPanel(rows, cols) {
    override lazy val peer = new javax.swing.JPanel(new java.awt.GridLayout(rows, cols)) with SuperMixin
  }
  
  // Image saver
  
  // Image dialog filter
  private object pngFilter extends javax.swing.filechooser.FileFilter {	  
    def accept(f: File): Boolean = {
      if (f ne null) {
        if (f.isDirectory)
          return true
        if (f.getName.endsWith(".png"))
    	  return true
      }
      return false
    }
    def getDescription(): String = "Portable Network Graphics (*.png)"
  }
    
  // Image view
  private[CharacterCubePanel] final class ImageView(private var image: BufferedImage) extends Component {
	size = new Dimension(image.getWidth, image.getHeight)
	preferredSize = size
	maximumSize = size
	minimumSize = size
	
	override def paintComponent(g2d: Graphics2D) {               
      super.paintComponent(g2d)  
      if (image != null)
        g2d.drawImage(image, 0, 0, image.getWidth, image.getHeight, null)       
	}
	
	private[CharacterCubePanel] def clear {
	  image = null
	}
  }
  
  import java.awt.event.WindowEvent
  import java.awt.event.WindowEvent._
  import javax.swing.JDialog
  
  // Image dialog
  private[CharacterCubePanel] class ImageDialog(owner: Frame) extends Dialog(owner) {
	  
    override lazy val peer = new JDialog(owner.peer) with InterfaceMixin {
    	
      override protected def processWindowEvent(e: WindowEvent) {
        super.processWindowEvent(e)
        if (e.getID == WINDOW_CLOSING)
          closeOperation
        else if (e.getID == WINDOW_CLOSED)
          closeOperation
      }
    } 
  }
  
}
