package org.interactivemesh.scala.j3d.samples.charcube

// Java
import java.awt.{Color, Dimension, Font, GradientPaint, Graphics2D, 
  GraphicsConfigTemplate, GraphicsDevice}
import java.awt.image.BufferedImage

import java.util.{Enumeration => JEnumeration}

// Java 3D
import javax.media.j3d.{Alpha, Appearance, Background, Behavior, BoundingBox,
  BoundingSphere, Bounds, BranchGroup, Canvas3D, DirectionalLight, GeometryArray,
  GraphicsConfigTemplate3D, Group, ImageComponent, ImageComponent2D, Locale, 
  Material, OrderedGroup, PhysicalBody, PhysicalEnvironment, PickInfo, 
  PolygonAttributes, QuadArray, RestrictedAccessException, RotationInterpolator,
  Screen3D, Shape3D, Switch, Transform3D, TransformGroup, TransparencyAttributes,
  View, ViewPlatform, VirtualUniverse, WakeupCriterion, WakeupOnBehaviorPost}

import javax.vecmath.{AxisAngle4d, Color3f, Point3d, Vector3d}

import com.sun.j3d.utils.pickfast.PickCanvas

// AWTShapeExtruder API 3.0, see: http://www.interactivemesh.org/testspace/awtshapeextruder.html
import com.interactivemesh.j3d.community.utils.geometry.{AWTShapeExtruder, AWTShapeExtrusion, String3D}

// OrbitBehaviorInterim API 2.1, see http://www.interactivemesh.org/testspace/orbitbehavior.html
import com.interactivemesh.j3d.community.utils.navigation.orbit.OrbitBehaviorInterim

/*
 * CharacterCubeUniverse.scala
 *
 * Version: 1.3
 * Date: 2011/05/26
 *
 * Copyright (c) 2010-2011
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 *
 * Please create your own implementation.
 * This source code is provided "AS IS", without warranty of any kind.
 * You are allowed to copy and use all lines you like of this source code
 * without any copyright notice,
 * but you may not modify, compile, or distribute this 'CharacterCubeUniverse.scala'.
 * 
 */

private final class CharacterCubeUniverse(panel: CharacterCubePanel, gd: GraphicsDevice) {
  
  private val globalBounds = new BoundingSphere(new Point3d, java.lang.Double.MAX_VALUE)

  private val bgColor0 = new Color(0, 102, 204) // RGB 0.0f, 0.4f, 0.8f, HSB 209, 100,  80
  private val bgColor1 = new Color(0, 153, 255) // RGB 0.0f, 0.6f, 1.0f, HSB 203, 100, 100
  
  private val rotationStartTime = System.currentTimeMillis

  
  // Viewpoints
  private[charcube] object ViewPoint extends Enumeration {
    type ViewPoint = Value
    val Front, Back, Left, Right, Home, Center = Value
  }
  
  // Vantage points
  private abstract class VantagePoint {
    val posParallel = new Point3d
    val posPerspective = new Point3d
    val lookAt = new Point3d(0, 0, 0)
    val upVector = new Vector3d(0, 1, 0)
    val rotCenter = new Point3d(0, 0, 0)
    
    def applyTo(navigator: OrbitBehaviorInterim) {
      if (navigator.getProjectionMode == View.PARALLEL_PROJECTION)
        navigator.setViewingTransform(posParallel, lookAt, upVector, rotCenter)
      else
        navigator.setViewingTransform(posPerspective, lookAt, upVector, rotCenter)
    }
  }
  
  import ViewPoint._
  
  // Map : ViewPoint -> VantagePoint
  private val vantagePoints = Map(
    Front -> new VantagePoint {
      posParallel.set(0, 0, 1600)
      posPerspective.set(0, 0, 1900)
    },
    Back -> new VantagePoint {
      posParallel.set(0, 0, -1600)
      posPerspective.set(0, 0, -1900)
    },
    Left -> new VantagePoint {
      posParallel.set(-1600, 0, 0)
      posPerspective.set(-1900, 0, 0)
    },
    Right -> new VantagePoint {
      posParallel.set(1600, 0, 0)
      posPerspective.set(1900, 0, 0)
    },
    Home -> new VantagePoint {
      posParallel.set(1600, 1000, 1600)
      posPerspective.set(1600, 1000, 1600)
    },
    Center -> new VantagePoint {
      posParallel.set(0, 0, 0)
      posPerspective.set(0, 0, 0)
      lookAt.set(0, 0, -1)
    }
  )
  
  //
  // SuperStructure
  //
  
  private val vu = new VirtualUniverse
  private val locale = new Locale(vu)

  //
  // Viewing
  //
  
  private val view = new View {
    setPhysicalBody(new PhysicalBody)
    setPhysicalEnvironment(new PhysicalEnvironment)
  }  
  
  private val orbitBehavior = new OrbitBehaviorInterim(OrbitBehaviorInterim.REVERSE_ALL) {
    setSchedulingBounds(globalBounds)
    setClippingEnabled(true)
    setVpView(CharacterCubeUniverse.this.view) // due to 'view'-naming collision in OrbitBehaviorInterim
  }
 
  private val canvas3d = new Canvas3D(gd.getBestConfiguration(new GraphicsConfigTemplate3D)) {
    // Less 'flickering' 
    setBackground(bgColor0)

    orbitBehavior.setAWTComponent(this)   
  }
  
  // View renders into canvas3D
  view.addCanvas3D(canvas3d)
  
  // ImageRenderer : offCanvas3D
  private val gcfgOff = gd.getBestConfiguration(new GraphicsConfigTemplate3D {
    setDoubleBuffer(GraphicsConfigTemplate.UNNECESSARY)
    setStereo(GraphicsConfigTemplate.UNNECESSARY)
  })
  view.addCanvas3D(offCanvas3D)

  // Properties
  private val c3dProps = canvas3d.queryProperties
  // Max offscreen buffer size
  private val maxTexSize = new Dimension(
    c3dProps.get("textureWidthMax").asInstanceOf[Int], 
    c3dProps.get("textureHeightMax").asInstanceOf[Int]
  )

  private val phyScreenWidth = canvas3d.getScreen3D.getPhysicalScreenWidth
  
  //
  // BranchGraphs
  //
  
  private val enviBranch = new BranchGroup
  private val sceneBranch = new BranchGroup {
	setCapability(Group.ALLOW_CHILDREN_EXTEND)
  }
  private val viewBranch = new BranchGroup
  
  // PickCanvas operates on the canvas3D
  private val pickCanvas = new PickCanvas(canvas3d, sceneBranch) {
    setMode(PickInfo.PICK_GEOMETRY)
    setFlags(PickInfo.NODE)
    setTolerance(4.0f)
  }
  
  // View branch
  
  viewBranch.addChild(
    new TransformGroup {
      setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE) 
      
      val vP = new ViewPlatform
      
      view.attachViewPlatform(vP)    
      orbitBehavior.setViewingTransformGroup(this)
      
      addChild(vP)
      addChild(orbitBehavior)
      addChild( new DirectionalLight { setInfluencingBounds(globalBounds) } )
    }
  )  
  
  // Environment branch  
  
  private val background = new Background {
    setCapability(Background.ALLOW_IMAGE_WRITE)
    setApplicationBounds(globalBounds)
    setColor(new Color3f(bgColor0))
  }
  
  enviBranch.addChild(background)
  enviBranch.addChild(speedBehavior)
  enviBranch.addChild(vantagePointBehavior)

  // Scene branch
  
  // String3D / Extruder  (105572 triangles)
  private object string3D extends String3D {
	  
    // Extrusion
    private val extend = 1.0f
    private val depth = 20.0f
    private val cut = depth/10.0f

    private val extrPath = new java.awt.geom.GeneralPath // GeneralPath is final
    extrPath.moveTo(0.0f, 0.0f)
    extrPath.lineTo(cut, extend)
    extrPath.lineTo(depth-cut, extend)
    extrPath.lineTo(depth, 0.0f)
	  	  
    setFont(new Font("Lucida Sans", Font.PLAIN, 100))
    setExtruder(new AWTShapeExtruder(0.05, new AWTShapeExtrusion(extrPath), Math.toRadians(24)))
    setAlignment(String3D.Alignment.CENTER)
    setPath(String3D.Path.RIGHT)
  }
   
  // Characters' Appearances for each cube face
  
  private val dim = 5
  
  private val charAppears = new Array[Appearance](dim*dim)
  fill(charAppears)
  
  private def fill(appears: Array[Appearance]) {
    var k = 0
    var direction = 1
    var hue = 0f
    val step = 1.0f / (dim*dim)

    for (i <- 0 until dim) {
      for (j <- 0 until dim) {
        appears(k) = new Appearance
        appears(k).setMaterial(
          new Material {
            setAmbientColor(0.0f, 0.0f, 0.0f)
            setDiffuseColor(new Color3f(new Color(Color.HSBtoRGB(hue, 0.85f, 0.7f))))
            setEmissiveColor(new Color3f(new Color(Color.HSBtoRGB(hue, 0.85f, 0.3f))))
            setSpecularColor(0.0f, 0.0f, 0.0f)
            setShininess(64f)
          }
        )       
        
        hue += step * direction
        k += 1
      }

      direction *= (-1)
      if (direction < 0) hue += step * (dim-1)
      else               hue += step * (dim+1)
    }
  }
  
  // Collects all characters of all faces
  private val charGroup = new Group

  private var targetTG: TransformGroup = null

  // Arrows
  private val arrows = Array(8608, 8646, 8673, 8611, 8651,
                             8613, 8614, 8670, 8632, 8676,
                             8618, 8605, 8620, 8621, 8668,
                             8679, 8653, 8661, 8666, 8665,
                             8634, 8629, 8630, 8625, 8623)
  targetTG = new TransformGroup {
    setTransform(new Transform3D {
      setRotation(new AxisAngle4d(0, 1, 0, Math.toRadians(180)))
      setTranslation(new Vector3d(700-90, 50, -50))                // Back
    })
  }
  addCharacters(arrows, targetTG)
  charGroup.addChild(targetTG)
  
  // Dingbats
  private val bats = Array( 9986,  9990,  9991,  9992,  9988,  
                            9998,  9997, 10001, 10002,  9993,
                           10006, 10007, 10070, 10012, 10004,
                           10051, 10053, 10021, 10052, 10057,
                           10102, 10113, 10104, 10115, 10106)
  targetTG = new TransformGroup {
    setTransform(new Transform3D {
      setTranslation(new Vector3d(90, 50, 700+50))                // Front
    })
  }
  addCharacters(bats, targetTG)
  charGroup.addChild(targetTG)
  
  // Latin
  private val latins = Array(  65,  66,  77,  89,  90,
                               33,  37,  38,  35,  63,
                              123,  43,  64,  47,  93,
                              167, 181, 164, 169, 182,
                              283, 339, 294, 373, 367)
  targetTG = new TransformGroup {
    setTransform(new Transform3D {
      setRotation(new AxisAngle4d(0, 1, 0, Math.toRadians(90)))
      setTranslation(new Vector3d(700+50, 50, 610))                // Right
    })
  }
  addCharacters(latins, targetTG)
  charGroup.addChild(targetTG)
  
  // Mathematical Operators
  private val mathOps = Array(8704, 8707, 8709, 8721, 8748,
                              8771, 8777, 8781, 8763, 8783,
                              8811, 8807, 8812, 8804, 8810,
                              8834, 8844, 8833, 8847, 8837,
                              8853, 8856, 8857, 8860, 8862)
  targetTG = new TransformGroup {
    setTransform(new Transform3D {
      setTranslation(new Vector3d(-50, 50, 90))                    // Left
      setRotation(new AxisAngle4d(0, 1, 0, Math.toRadians(-90)))
    })
  }
  addCharacters(mathOps, targetTG)
  charGroup.addChild(targetTG)
  
  private def addCharacters(characters: Array[Int], targetTG: TransformGroup) {

    val t3d = new Transform3D
    val translation = new Vector3d(0, 0, 0)

    var k = 0
    for (i <- 0 until dim) {
      for (j <- 0 until dim) {
        targetTG.addChild(
          new TransformGroup {
            t3d.setTranslation(translation)
            setTransform(t3d)
            addChild(new Unicode3D(
              string3D.getStringGeometry(new String(characters, k, 1)), 
              charAppears(k), 
              characters(k))
            )
          }
        )

        k += 1
        translation.x += 130
      }

      translation.x = 0
      translation.y += 130
    }
  }
  
  // Cube shape
  
  private object cubeShape3D extends Shape3D {
	  
    private[CharacterCubeUniverse] val cubeTransAttr = new TransparencyAttributes {
      setCapability(TransparencyAttributes.ALLOW_VALUE_WRITE)
      setTransparencyMode(TransparencyAttributes.NICEST)
      setTransparency(0.3f) // Initial value
    }
	  	  
    import GeometryArray._
    
    setGeometry(
      CubeGeometry.create(COORDINATES | NORMALS | COLOR_3, 700)
    )
        
    setAppearance(
      new Appearance {
        setMaterial(new Material {
    	  setColorTarget(Material.DIFFUSE)  // default
    	  setAmbientColor(0, 0, 0)
    	  setSpecularColor(0, 0, 0)
        })
        setPolygonAttributes(new PolygonAttributes {
          setBackFaceNormalFlip(true)
          setCullFace(PolygonAttributes.CULL_NONE)
        })
        setTransparencyAttributes(cubeTransAttr)
      }
    )
  }
  
  // Hide fully transparent cube
  private val cubeSwitch = new Switch {
    setCapability(Switch.ALLOW_SWITCH_READ)
    setCapability(Switch.ALLOW_SWITCH_WRITE)
    setWhichChild(Switch.CHILD_ALL)
    
    addChild(cubeShape3D)
  }

  // Rotation

  private val rotationGroup = new TransformGroup {
    setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE)
    // Center cube at (0,0,0)
    addChild(new TransformGroup { 
      setTransform(new Transform3D { 
        setTranslation(new Vector3d(-350, -350, -350))
      })
      // Order opaque/transparency shapes
      addChild(new OrderedGroup {
        addChild(charGroup)
        addChild(cubeSwitch)
      })
    })
  }
  
  sceneBranch.addChild(rotationGroup)
  sceneBranch.addChild(rotInterpolatorWithFPS)
  
  // Set live
  
  locale.addBranchGraph(sceneBranch)
  locale.addBranchGraph(viewBranch)
  locale.addBranchGraph(enviBranch)
  
  // Setup navigator
  var sceneRadius = 1.0d

  val bounds = sceneBranch.getBounds
  var sphereBounds = new BoundingSphere
  if (!bounds.isEmpty) {
    if (sphereBounds.isInstanceOf[BoundingSphere])
      sphereBounds = bounds.asInstanceOf[BoundingSphere]
    else
      sphereBounds = new BoundingSphere(bounds)
  }
        
  sceneRadius = sphereBounds.getRadius

  orbitBehavior.setTransFactors(sceneRadius/8.0f, sceneRadius/8.0f)
  orbitBehavior.setZoomFactor(sceneRadius/8.0f)
  orbitBehavior.setRotFactors(0.25f, 0.25f)

  orbitBehavior.setClippingBounds(sphereBounds)

  // Initial viewpoint (vantagePointBehavior not active yet)
  vantagePoints.get(Home).get.applyTo(orbitBehavior)
  
  //
  // Universe interaction
  //
  
  private[charcube] def canvas3D: Canvas3D = canvas3d

  private[charcube] def setupBgImage(width: Int, height: Int): Unit = 
    setupBgImage(background, width, height, false)
  
  private[charcube] def maxOffscreenBufferSize: Dimension =	 maxTexSize

  private[charcube] def renderToImage(width: Int, height: Int, transBackground: Boolean): Unit = {
	    	
	// javax.media.j3d.CanvasViewCache.computeViewPlatformScale :
	//   windowScale = physicalWindowWidth / physicalScreenWidth
	//   viewPlatformScale = windowScale * screenScale
	// 
	// canvas3D and offCanvas3D have a different Screen3D object
	//   in offCanvas3D: 1 = windowScaleOff = physicalWindowWidthOff / physicalScreenWidthOff
	//   to be in sync with windowScale in canvas3D multiply current screenScale
	//   with ratio of new image ph.width and physicalScreenWidth
	
    val isParallel = ( (view.getProjectionPolicy == View.PARALLEL_PROJECTION) )
    val screenScale = view.getScreenScale

    if (isParallel)
      view.setScreenScale(screenScale * (width*offCanvas3D.METERS_PER_PIXEL / phyScreenWidth) )

    setupBgImage(background, width, height, transBackground)
	
    offCanvas3D.render(width, height, transBackground)
	
    setupBgImage(background, canvas3D.getWidth, canvas3D.getHeight, false)
	
    if (isParallel)
      view.setScreenScale(screenScale)
  }

  private[charcube] def projectionMode(mode: Int): Unit = 
	orbitBehavior.setProjectionMode(mode)

  private[charcube] def vantagePoint(v: ViewPoint): Unit = 
    vantagePointBehavior.vantagePoint(vantagePoints.get(v).get)

  private[charcube] def cubeTransparency(t: Int): Unit = {
    var transparency = t
    if (transparency > 100)
      transparency = 100
    else if (transparency < 0)
      transparency = 0
    
    var value = transparency/100.0f

    if (value > 0.98) {
      value = 1.0f
      cubeSwitch.setWhichChild(Switch.CHILD_NONE)
    }
    else if (cubeSwitch.getWhichChild != Switch.CHILD_ALL) {
      cubeSwitch.setWhichChild(Switch.CHILD_ALL)
    }
    
    cubeShape3D.cubeTransAttr.setTransparency(value)
  }
  
  // Rotation speed and direction
  private[charcube] def resetRotation: Unit = rotationAlpha.pause(rotationStartTime)
  private[charcube] def rotationSpeed(s: Int): Unit = speedBehavior.speed(s)
  
  private[charcube] def shapePicked(x: Int, y: Int): Unit = changeRotationCenter(x, y)
  
  private[charcube] def closeUniverse: Unit = { 
    view.removeAllCanvas3Ds
    view.attachViewPlatform(null)
    vu.removeAllLocales
  }
  
  //
  // Nested classes, defs, behaviors
  //
  
  // Unicode3D shape
  private final class Unicode3D(
      geom: GeometryArray, appear: Appearance, val codepoint: Int) 
  	    extends Shape3D(geom, appear)
  
  // Rotation center
  private def changeRotationCenter(pickX: Int, pickY: Int) {
    
    pickCanvas.setShapeLocation(pickX, pickY)

    val pickInfo = pickCanvas.pickClosest    
    if (pickInfo eq null) 
      return
         
    val locToVWord = new Transform3D    
    val rotationPoint = new Point3d    
    
    val pickedShape3D = pickInfo.getNode.asInstanceOf[Shape3D]    
    pickedShape3D.getLocalToVworld(locToVWord)    
       
    pickedShape3D.getBounds match {
      case b: BoundingBox =>
        val lower = new Point3d
        val upper = new Point3d
        b.getLower(lower)
        b.getUpper(upper)
        rotationPoint.set(lower.x + (upper.x - lower.x)/2,
                          lower.y + (upper.y - lower.y)/2,
                          lower.z + (upper.z - lower.z)/2)
    	
      case b: BoundingSphere =>
        b.getCenter(rotationPoint)
    	
      case b: Bounds =>
        new BoundingSphere(b).getCenter(rotationPoint)
    }
    
    if (pickedShape3D.isInstanceOf[Unicode3D])
      panel.unicodeCodePointSelected( pickedShape3D.asInstanceOf[Unicode3D].codepoint )

    locToVWord.transform(rotationPoint)
    orbitBehavior.setRotationCenter(rotationPoint, true) // isLookAtRotCenter    
  }  
  
  // Image for Background node
  private def setupBgImage(bg: Background, width: Int, height: Int, transparent: Boolean) {   	
    if (transparent) {
      bg.setImage(null)
    }
    else {
      val gradientImage = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB)
      val g2d = gradientImage.getGraphics.asInstanceOf[Graphics2D]
	
      val splitHeight = (height/5f).asInstanceOf[Int] // yUp !
	
      var gp = new GradientPaint(0, 0,           bgColor0,
                                 0, splitHeight, bgColor1)
      g2d.setPaint(gp)
      g2d.fillRect(0, 0, width, splitHeight)
	
      gp = new GradientPaint(0, splitHeight, bgColor1,
                             0, height,      bgColor0)
      g2d.setPaint(gp)
      g2d.fillRect(0, splitHeight, width, height)
	
      g2d.dispose
	
      bg.setImage(new ImageComponent2D(ImageComponent.FORMAT_RGB, gradientImage, true, true))
    }
  }

  // Sets vantage point in behavior scheduler
  private object vantagePointBehavior extends Behavior {
   
    setSchedulingBounds(globalBounds)

    private val APPLY_VP = 1
    private val post = new WakeupOnBehaviorPost(this, APPLY_VP)

    private var vP: VantagePoint = null

    private[CharacterCubeUniverse] def vantagePoint(vp: VantagePoint) {
      vP = vp
      postId(APPLY_VP)
    }

    override def initialize = wakeupOn(post)
	
    override def processStimulus(criteria: JEnumeration[_]) { // renamed java.util.Enumeration
      if (vP != null) {
        if (rotationAlpha.isPaused)
          rotationAlpha.pause(rotationStartTime)
        vP.applyTo(orbitBehavior)
      }
      vP = null
      wakeupOn(post)
	}
  }
  
  private object rotationAlpha extends Alpha {
	  
    setLoopCount(-1)
    setIncreasingAlphaDuration(2000 * 47)                  // start speed = min speed
    setStartTime(rotationStartTime)                        // start
    pause(rotationStartTime)                               // pause at start time
    
    private[CharacterCubeUniverse] def rotationSpeed(s: Int) {
      var speed = s
      if (speed > 99)     speed = 99
      else if (speed < 1) speed = 1

      // Loop duration determines rotation speed
      var duration = 0L
      var pauseTime = 0L
      var pauseValue = 0f

      if (speed > 52) {
    	
    	rotInterpolatorWithFPS.enableFPS(true)

        // isPaused
        if (rotationAlpha.isPaused) {
          if (this.getMode == Alpha.INCREASING_ENABLE) {
            rotationAlpha.resume
            return
          }
          pauseTime = rotationAlpha.getPauseTime
        }
        // Pause
        else {
          pauseTime = System.currentTimeMillis
          this.pause(pauseTime)
        }

        pauseValue = this.value

        if (this.getMode != Alpha.INCREASING_ENABLE)
          this.setMode(Alpha.INCREASING_ENABLE)

        // New IncreasingAlphaDuration
//      duration = (long)1000 *((100 - speed))                 // [47, 1] non linear
        duration = (2000 * 47.0/(speed - 52) ).asInstanceOf[Long]    // [47, 1] linear

        // Offset according to this's pauseValue and the new IncreasingAlphaDuration
        val resumeOffsetTime = (pauseValue * duration).asInstanceOf[Long]

        this.setIncreasingAlphaDuration(duration)

        // Resume
        // Alpha source code: newStartTime = oldStartTime + resumeTime - pauseTime
        // Start immediately and adapt new duration:
        //   => System.currentTimeMillis - resumeOffsetTime = oldStartTime + resumeTime - pauseTime
        //   => resumeTime = System.currentTimeMillis - resumeOffsetTime - oldStartTime + pauseTime

        this.resume(System.currentTimeMillis - resumeOffsetTime - this.getStartTime + pauseTime)
      }
      else if (speed < 48) {
    	
        rotInterpolatorWithFPS.enableFPS(true)

        // isPaused
        if (this.isPaused) {
          if (this.getMode == Alpha.DECREASING_ENABLE) {
            this.resume
            return
          }
          pauseTime = this.getPauseTime
        }
        // Pause
        else {
          pauseTime = System.currentTimeMillis
          this.pause(pauseTime)
        }

        pauseValue = this.value

        if (this.getMode != Alpha.DECREASING_ENABLE)
          this.setMode(Alpha.DECREASING_ENABLE)

        // New DecreasingAlphaDuration
//      duration = (long)(1000 * speed)                        // [47, 1] non linear
        duration = (2000 * 47.0/(48 - speed)).asInstanceOf[Long]    // [47, 1] linear

        // Offset according to pauseValue and the new DecreasingAlphaDuration
        val resumeOffsetTime = ((1-pauseValue) * duration).asInstanceOf[Long]

        this.setDecreasingAlphaDuration(duration)

        // Resume
        this.resume(System.currentTimeMillis - resumeOffsetTime - this.getStartTime + pauseTime)
      }
      else {
        if (!this.isPaused)
          this.pause(System.currentTimeMillis)
        
        rotInterpolatorWithFPS.enableFPS(false)
      }
    }
  }

  // RotationInterpolator inclusive frames per second calculation and printing
  private object rotInterpolatorWithFPS extends RotationInterpolator(rotationAlpha, rotationGroup) {
        
	setSchedulingBounds(globalBounds)
	  
    private var startTime = System.nanoTime      
    private var frameCounter  = 0     
    private var elapsedFrames =  10
        
    private var isUpdateFPS = false
        
    private[CharacterCubeUniverse] def enableFPS(enable: Boolean) {        	
      isUpdateFPS = enable
      if (enable && !isUpdateFPS) {
        frameCounter = 0
        elapsedFrames = 10
        startTime = System.nanoTime
      }
    }
        
    override def processStimulus(criteria: JEnumeration[_]) { // renamed java.util.Enumeration                            
      if (isUpdateFPS) {
        frameCounter += 1
	            
        if (frameCounter >= elapsedFrames) {
	                
          val stopTime = System.nanoTime
          val frameDuration = ((stopTime - startTime)/frameCounter).asInstanceOf[Double]
	                
          val fps = 1000000000d / frameDuration
	                
          startTime = stopTime
	
          frameCounter = 0
          elapsedFrames = Math.max(1, ((fps + 0.5)/4)).asInstanceOf[Int] // 4 times per second
	                
          panel.updateFPS(fps.asInstanceOf[Int])
        }
      }
      else 
    	panel.updateFPS(0)
                      
      // RotationInterpolator
      super.processStimulus(criteria)                          
    }       
  } 
  
  // Rotation speed update
  private object speedBehavior extends Behavior {
        
	setSchedulingBounds(globalBounds)

    private val speedPost = new WakeupOnBehaviorPost(this, 0)

    private[CharacterCubeUniverse] def speed(s: Int) = postId(s)

    override def initialize = wakeupOn(speedPost)
    
    override def processStimulus(criteria: JEnumeration[_]) { // renamed java.util.Enumeration
      while (criteria.hasMoreElements) {   	  
        criteria.nextElement match {        	
          case w: WakeupOnBehaviorPost =>
            rotationAlpha.rotationSpeed(w.getTriggeringPostId)
        }
      }      
      wakeupOn(speedPost)
    }
  }
 
  // Ofscreen Canvas3D for 'Rendering to Image'
  private object offCanvas3D extends Canvas3D(gcfgOff, true) {
	  
    import BufferedImage._
    import ImageComponent._
	
    // Screen3D :
    // Assume a default of 90 DPI: 90 pix/inch = 1/90 inch/pix = 0.0254/90 meter/pix
    private[CharacterCubeUniverse] val METERS_PER_PIXEL = 0.0254 / 90.0
    private val screen3D = this.getScreen3D	
    
    // Reused image if of same size and type
    private var image: BufferedImage = new BufferedImage(10, 10, TYPE_INT_RGB)
		
    private[CharacterCubeUniverse] def render(width: Int, height: Int, transBg: Boolean) {
		
      this.waitForOffScreenRendering
                   
      // Compute physical dimensions of the screen     
      screen3D.setSize(width, height)
      screen3D.setPhysicalScreenWidth(width * METERS_PER_PIXEL)
      screen3D.setPhysicalScreenHeight(height * METERS_PER_PIXEL)
                   
      val bImageType = if (transBg) TYPE_INT_ARGB else TYPE_INT_RGB
      val iCompType = if (transBg) FORMAT_RGBA else FORMAT_RGB
      
      if (image.getWidth != width || image.getHeight != height || image.getType != bImageType) {
    	image.flush
        image = new BufferedImage(width, height, bImageType)
    	System.gc
      }

      // OffScreenBuffer: byReference & yUp 
      val imageComp = new ImageComponent2D(iCompType, image, true, true) {
        setCapability(ALLOW_IMAGE_WRITE)
      }
        
      try {
        this.setOffScreenBuffer(imageComp)               
        this.renderOffScreenBuffer
      }
      catch {
    	case e: RestrictedAccessException => render(width, height, transBg) 
    	// TODO Risk of endless cycle ?
      }
    }
	
    override def postSwap {
      this.setOffScreenBuffer(null)
      flipImageY(image)
      panel.showImageDialog(image) 
    }
    
    private def flipImageY(bImage: BufferedImage) {
		
      val wRaster = bImage.getRaster
		
      // If null, an array of appropriate type and size will be allocated
      var rowTop: Object = null
      var rowBot: Object = null
	    
      val width = bImage.getWidth
      val height = bImage.getHeight
      val hIndex = height - 1
      val halfHeight = height/2
	      
      // Switch first and last, first-1 and last-1, etc.
      for (i <- 0 until halfHeight) {
        // getDataElements(int x, int y, int w, int h, Object outData)
        rowTop = wRaster.getDataElements(0, i, width, 1, rowTop)
        rowBot = wRaster.getDataElements(0, hIndex - i, width, 1, rowBot)
        // setDataElements(int x, int y, int w, int h, Object inData)
        wRaster.setDataElements(0, i, width, 1, rowBot)
        wRaster.setDataElements(0, hIndex - i, width, 1, rowTop)
      }
	}

  }  
  
  // Cube factory
  private object CubeGeometry {
	  
    private val coords = Array(
      0.0f, 0.0f, 1.0f,   1.0f, 0.0f, 1.0f,   1.0f, 1.0f, 1.0f,   0.0f, 1.0f, 1.0f,   // back    +Z
      1.0f, 0.0f, 0.0f,   0.0f, 0.0f, 0.0f,   0.0f, 1.0f, 0.0f,   1.0f, 1.0f, 0.0f,   // front   -Z   
      1.0f, 0.0f, 1.0f,   1.0f, 0.0f, 0.0f,   1.0f, 1.0f, 0.0f,   1.0f, 1.0f, 1.0f,   // right   +X
      0.0f, 0.0f, 0.0f,   0.0f, 0.0f, 1.0f,   0.0f, 1.0f, 1.0f,   0.0f, 1.0f, 0.0f,   // left    -X
      0.0f, 1.0f, 1.0f,   1.0f, 1.0f, 1.0f,   1.0f, 1.0f, 0.0f,   0.0f, 1.0f, 0.0f,   // top     +Y
      0.0f, 0.0f, 1.0f,   0.0f, 0.0f, 0.0f,   1.0f, 0.0f, 0.0f,   1.0f, 0.0f, 1.0f    // bottom  -Y
    )

    private val normals = Array(
       0.0f,  0.0f,  1.0f,   0.0f,  0.0f,  1.0f,   0.0f,  0.0f,  1.0f,   0.0f,  0.0f,  1.0f,  // +Z
       0.0f,  0.0f, -1.0f,   0.0f,  0.0f, -1.0f,   0.0f,  0.0f, -1.0f,   0.0f,  0.0f, -1.0f,  // -Z
       1.0f,  0.0f,  0.0f,   1.0f,  0.0f,  0.0f,   1.0f,  0.0f,  0.0f,   1.0f,  0.0f,  0.0f,  // +X
      -1.0f,  0.0f,  0.0f,  -1.0f,  0.0f,  0.0f,  -1.0f,  0.0f,  0.0f,  -1.0f,  0.0f,  0.0f,  // -X
       0.0f,  1.0f,  0.0f,   0.0f,  1.0f,  0.0f,   0.0f,  1.0f,  0.0f,   0.0f,  1.0f,  0.0f,  // +Y
       0.0f, -1.0f,  0.0f,   0.0f, -1.0f,  0.0f,   0.0f, -1.0f,  0.0f,   0.0f, -1.0f,  0.0f   // -Y
    )

    private val colors = Array(
      0.0f, 0.0f, 1.0f,   1.0f, 0.0f, 1.0f,   1.0f, 1.0f, 1.0f,   0.0f, 1.0f, 1.0f,           // +Z
      1.0f, 0.0f, 0.0f,   0.0f, 0.0f, 0.0f,   0.0f, 1.0f, 0.0f,   1.0f, 1.0f, 0.0f,           // -Z
      1.0f, 0.0f, 1.0f,   1.0f, 0.0f, 0.0f,   1.0f, 1.0f, 0.0f,   1.0f, 1.0f, 1.0f,           // +X
      0.0f, 0.0f, 0.0f,   0.0f, 0.0f, 1.0f,   0.0f, 1.0f, 1.0f,   0.0f, 1.0f, 0.0f,           // -X
      0.0f, 1.0f, 1.0f,   1.0f, 1.0f, 1.0f,   1.0f, 1.0f, 0.0f,   0.0f, 1.0f, 0.0f,           // +Y
      0.0f, 0.0f, 1.0f,   0.0f, 0.0f, 0.0f,   1.0f, 0.0f, 0.0f,   1.0f, 0.0f, 1.0f            // -Y
    )
  
    private[CharacterCubeUniverse] def create(vertexFormat: Int, cubeSize: Float): GeometryArray = {

      val geometry = new QuadArray(24, vertexFormat)

      if (cubeSize != 1.0f) {
    	val length = 24*3
    	val sizeCoords = new Array[Float](length)
        for (i <- 0 until length) {
          if (coords(i) > 0)
            sizeCoords(i) = cubeSize
        }
    	geometry.setCoordinates(0, sizeCoords)
      }
      else
        geometry.setCoordinates(0, coords)
      
      if ((vertexFormat & GeometryArray.NORMALS) != 0)
        geometry.setNormals(0, normals)
      if ((vertexFormat & GeometryArray.COLOR_3) != 0)
        geometry.setColors(0, colors)

      return geometry
    } 
  }
  
  // Java 3D properties
  private[charcube] def printJava3DProps {

    val vuProps = VirtualUniverse.getProperties

    println("Java 3D Version  =  " + vuProps.get("j3d.version"))
    println("Java 3D Renderer  =  " + vuProps.get("j3d.renderer"))
    println("Java 3D Pipeline  =  " + vuProps.get("j3d.pipeline"))
    println("------------------------------------------------------------------------")

    if (c3dProps ne null) {
      println("Native Version  =  " + c3dProps.get("native.version"))
      println("Native GLSL  =  " + c3dProps.get("shadingLanguageGLSL"))
      println("Native Vendor  =  " + c3dProps.get("native.vendor"))
      println("Native Renderer  =  " + c3dProps.get("native.renderer"))
      println("------------------------------------------------------------------------")
      println("Max Texture Width  =  " + maxTexSize.width)
      println("Max Texture Height  =  " + maxTexSize.height)
      println("------------------------------------------------------------------------")
    }
    println("")
  }

}
